import ApiService from "@/services/ApiService";

const baseUrl = "config-impresiones";

/**
 * Obtener todas las configuraciones de impresión
 */
export const getAll = async (tipoImpresion = null, sucursalId = null) => {
  const params = {};
  if (tipoImpresion) {
    params.tipo_impresion = tipoImpresion;
  }
  if (sucursalId) {
    params.sucursal_id = sucursalId;
  }
  const response = await ApiService.get(`${baseUrl}/all`, { params });
  return response.data;
};

/**
 * Obtener la configuración activa por tipo
 */
export const getActiva = async (tipoImpresion = 'ticket', sucursalId = null) => {
  const params = { tipo_impresion: tipoImpresion };
  if (sucursalId) {
    params.sucursal_id = sucursalId;
  }
  const response = await ApiService.get(`${baseUrl}/activa`, { params });
  return response.data;
};

/**
 * Obtener estilos CSS de una configuración específica
 */
export const getEstilos = async (id) => {
  const response = await ApiService.get(`${baseUrl}/estilos/${id}`);
  return response.data;
};

/**
 * Crear una nueva configuración
 */
export const create = async (params) => {
  const response = await ApiService.post(`${baseUrl}/create`, params);
  return response.data;
};

/**
 * Obtener una configuración por ID
 */
export const getById = async (id) => {
  const response = await ApiService.get(`${baseUrl}/${id}`);
  return response.data;
};

/**
 * Actualizar una configuración
 */
export const update = async (params, id) => {
  const response = await ApiService.put(`${baseUrl}/${id}/edit`, params);
  return response.data;
};

/**
 * Eliminar una configuración
 */
export const deleteConfig = async (id) => {
  const response = await ApiService.delete(`${baseUrl}/${id}`);
  return response.data;
};

/**
 * Activar una configuración
 */
export const activar = async (id) => {
  const response = await ApiService.put(`${baseUrl}/${id}/activar`);
  return response.data;
};

/**
 * Establecer como predeterminada
 */
export const establecerPredeterminada = async (id) => {
  const response = await ApiService.put(`${baseUrl}/${id}/predeterminada`);
  return response.data;
};

/**
 * Duplicar una configuración
 */
export const duplicar = async (id) => {
  const response = await ApiService.post(`${baseUrl}/${id}/duplicar`);
  return response.data;
};

/**
 * Enviar prueba de impresión directa
 */
export const enviarPrueba = async (id) => {
  const response = await ApiService.post(`${baseUrl}/${id}/enviar-prueba`);
  return response.data;
};

/**
 * Enviar prueba de impresión de vale
 */
export const enviarPruebaVale = async (id) => {
  const response = await ApiService.post(`${baseUrl}/${id}/enviar-prueba-vale`);
  return response.data;
};

/**
 * Enviar prueba de impresión de etiquetas
 */
export const enviarPruebaEtiquetas = async (id) => {
  const response = await ApiService.post(`${baseUrl}/${id}/enviar-prueba-etiquetas`);
  return response.data;
};

/**
 * Configuración por defecto en caso de error
 */
export const getConfiguracionPorDefecto = () => {
  return {
    id: 0,
    nombre: 'Configuración por Defecto',
    tipo_impresion: 'ticket',
    ancho_papel: 80,
    alto_papel: 0,
    margen_superior: 0,
    margen_inferior: 0,
    margen_izquierdo: 0,
    margen_derecho: 0,
    fuente_principal: 'Arial',
    tamano_fuente_principal: 22,
    tamano_fuente_titulo: 28,
    tamano_fuente_subtitulo: 26,
    tamano_fuente_detalle: 20,
    color_texto: '#000000',
    peso_fuente: 'bold',
    interlineado: 1.5,
    alineacion_titulo: 'center',
    alineacion_texto: 'center',
    alineacion_empresa: 'center',
    alineacion_direccion: 'center',
    alineacion_folio_fecha: 'center',
    alineacion_total: 'center',
    alineacion_footer: 'center',
    mostrar_logo: true,
    mostrar_fecha: true,
    mostrar_hora: true,
    mostrar_cliente: true,
    mostrar_empleado: true,
    mostrar_observaciones: true,
    separador_lineas: '---',
    padding_celda: 2,
    ancho_tabla: 100,
    ancho_contenido_porcentaje: 88,
    ancho_cuerpo_porcentaje: 80,
    sucursal_id: null,
    es_activa: true,
    es_predeterminada: true
  };
};

/**
 * Generar estilos CSS basados en la configuración
 */
export const generarEstilosCSS = (configuracion) => {
  const anchoPapel = configuracion.ancho_papel + 'mm';
  const altoPapel = configuracion.alto_papel > 0 ? configuracion.alto_papel + 'mm' : 'auto';
  
  return `
    @page {
      size: ${anchoPapel} ${altoPapel};
      margin: ${configuracion.margen_superior}mm ${configuracion.margen_derecho}mm ${configuracion.margen_inferior}mm ${configuracion.margen_izquierdo}mm;
    }

    * {
      color: ${configuracion.color_texto} !important;
      -webkit-text-fill-color: ${configuracion.color_texto} !important;
      -webkit-print-color-adjust: exact !important;
      print-color-adjust: exact !important;
    }

    body {
      margin: 0;
      padding: 0;
      width: ${configuracion.ancho_cuerpo_porcentaje || 80}%;
      font-family: ${configuracion.fuente_principal}, sans-serif;
      font-weight: ${configuracion.peso_fuente};
      line-height: ${configuracion.interlineado};
    }

    .ticket-content {
      width: ${configuracion.ancho_contenido_porcentaje || 88}%;
      margin: 0;
      padding: 0;
    }

    .ticket-content table {
      width: ${configuracion.ancho_tabla}%;
      table-layout: fixed;
      border-collapse: collapse;
      margin: 0;
      padding: 0;
    }

    .ticket-content td,
    .ticket-content th,
    .ticket-content p,
    .ticket-content span,
    .ticket-content div {
      padding: ${configuracion.padding_celda}mm 0;
      font-size: ${configuracion.tamano_fuente_principal}px;
      line-height: ${configuracion.interlineado};
      font-weight: ${configuracion.peso_fuente};
      white-space: normal;
      word-wrap: break-word;
      color: ${configuracion.color_texto} !important;
      -webkit-text-fill-color: ${configuracion.color_texto} !important;
      // text-align: ${configuracion.alineacion_texto};
    }

    .ticket-content h1 {
      font-size: ${configuracion.tamano_fuente_titulo}px !important;
      text-align: ${configuracion.alineacion_titulo};
    }
    
    .ticket-content h2 {
      font-size: ${configuracion.tamano_fuente_subtitulo}px !important;
      text-align: ${configuracion.alineacion_titulo};
    }
    
    .ticket-content h3 {
      font-size: ${configuracion.tamano_fuente_subtitulo}px !important;
      text-align: ${configuracion.alineacion_titulo};
    }

    .ticket-content .detalle {
      font-size: ${configuracion.tamano_fuente_detalle}px !important;
    }
    .ticket-content .descripcion-producto {
      text-align: ${configuracion.alineacion_texto} !important;
    }

    /* Estilos específicos para elementos del ticket */
    .ticket-content .empresa-nombre {
      font-size: ${configuracion.tamano_fuente_titulo}px !important;
      font-weight: bold !important;
      text-align: ${configuracion.alineacion_empresa || 'center'} !important;
      margin: 0 0 0 0 !important;
      padding: 0 0 0 0 !important;
    }

    .ticket-content .empresa-tagline {
      font-size: ${configuracion.tamano_fuente_principal}px !important;
      text-align: ${configuracion.alineacion_empresa || 'center'} !important;
      margin: 0 0 0 0 !important;
      padding: 0 0 0 0 !important;
    }

    .ticket-content .empresa-direccion {
      font-size: ${configuracion.tamano_fuente_detalle}px !important;
      text-align: ${configuracion.alineacion_direccion || 'center'} !important;
      margin: 0 0 0 0 !important;
      padding: 0 0 0 0 !important;
    }

    .ticket-content .folio-fecha {
      font-size: ${configuracion.tamano_fuente_detalle}px !important;
      text-align: ${configuracion.alineacion_folio_fecha || 'center'} !important;
      margin: 0 0 0 0 !important;
      padding: 0 0 0 0 !important;
    }

    .ticket-content .total {
      font-size: ${configuracion.tamano_fuente_principal}px !important;
      font-weight: bold !important;
      text-align: ${configuracion.alineacion_total || 'center'} !important;
      margin: 0;
      padding: 0 0;
    }

    .ticket-content .footer {
      font-size: ${configuracion.tamano_fuente_detalle}px !important;
      font-weight: bold !important;
      text-align: ${configuracion.alineacion_footer || 'center'} !important;
      margin: 0;
      padding: 0 0;
    }

    .ticket-content img {
      width: 100%;
      max-width: none;
      height: auto;
      margin: 0 auto;
      display: block;
    }

    .separador {
      text-align: center;
      font-size: ${configuracion.tamano_fuente_detalle}px;
      margin: 0 0;
    }

    @media print {
      * {
        color: ${configuracion.color_texto} !important;
        -webkit-text-fill-color: ${configuracion.color_texto} !important;
        -webkit-print-color-adjust: exact !important;
        print-color-adjust: exact !important;
      }
      body {
        width: ${configuracion.ancho_cuerpo_porcentaje || 80}%;
      }
    }
  `;
};

/**
 * Imprimir contenido usando la configuración de la base de datos
 */
export const imprimirConConfiguracion = async (contentId, tipoImpresion = 'ticket', sucursalId = null) => {
  try {
    // Obtener configuración activa
    const response = await getActiva(tipoImpresion, sucursalId);
    const configuracion = response.success ? response.data : getConfiguracionPorDefecto();
    
    // Obtener el contenido a imprimir
    const printableContent = document.getElementById(contentId);
    if (!printableContent) {
      throw new Error('No se encontró el contenido a imprimir');
    }

    // Crear ventana de impresión
    const printWindow = window.open('', '_blank', 'height=600,width=800');
    
    // Generar estilos CSS
    const estilosCSS = generarEstilosCSS(configuracion);
    
    // Crear HTML completo
    const htmlContent = `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Ticket</title>
          <style>
            ${estilosCSS}
          </style>
        </head>
        <body>
          <div class="ticket-content">
            ${printableContent.innerHTML}
          </div>
          <script>
            window.onload = function() {
              window.print();
              setTimeout(function() {
                window.close();
              }, 500);
            }
          </script>
        </body>
      </html>
    `.trim();

    // Escribir contenido y cerrar documento
    printWindow.document.write(htmlContent);
    printWindow.document.close();

    return true;
  } catch (error) {
    console.error('Error al imprimir:', error);
    // Fallback a la función original si hay error
    imprimirOriginal(contentId);
    return false;
  }
};

/**
 * Obtener configuración de impresión directa
 */
export const getConfigImpresionDirecta = async () => {
  const response = await ApiService.get(`${baseUrl}/config-imprimir-directa`);
  return response.data;
};

/**
 * Imprimir ticket enviando al endpoint del backend
 */
export const imprimirTicketEndpoint = async (ventaId, params = {}) => {
  const response = await ApiService.post(`imprimir/ticket/${ventaId}`, {
    nombreImpresora: params.nombreImpresora || 'ticket',
    tipoTicket: params.tipoTicket || 'Venta',
    anchoPapel: params.anchoPapel || 600
  });
  return response.data;
};

/**
 * Imprimir vale enviando al endpoint del backend
 */
export const imprimirValeEndpoint = async (valeId, params = {}) => {
  const response = await ApiService.post(`imprimir/vale/${valeId}`, {
    nombreImpresora: params.nombreImpresora || 'ticket',
    anchoPapel: params.anchoPapel || 600,
    incluirImagen: params.incluirImagen !== undefined ? params.incluirImagen : true
  });
  return response.data;
};

/**
 * Imprimir ticket usando impresora directa configurada
 * Busca automáticamente la configuración por usuario y sucursal
 */
export const imprimirTicketDirecto = async (ventaId) => {
  const response = await ApiService.post(`${baseUrl}/imprimir-ticket/${ventaId}`);
  return response.data;
};

/**
 * Imprimir vale usando impresora directa configurada
 * Busca automáticamente la configuración por usuario y sucursal
 * @param {number} valeId - ID del vale
 * @param {string} qrData - Token del QR (opcional)
 */
export const imprimirValeDirecto = async (valeId, qrData = null) => {
  const payload = qrData ? { qrData } : {};
  const response = await ApiService.post(`${baseUrl}/imprimir-vale/${valeId}`, payload);
  return response.data;
};

/**
 * Función de impresión original como fallback
 */
export const imprimirOriginal = (contentId) => {
  const printableContent = document.getElementById(contentId);
  const printWindow = window.open('', '_blank', 'height=600,width=800');

  const htmlContent = `
    <!DOCTYPE html>
    <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Ticket</title>
        <style>
          @page {
            size: 80mm auto;
            margin: 0;
          }

          * {
            color: rgb(0, 0, 0) !important;
            -webkit-text-fill-color: rgb(0, 0, 0) !important;
            -webkit-print-color-adjust: exact !important;
            print-color-adjust: exact !important;
          }

          body {
            margin: 0;
            padding: 0;
            width: 88%;
            font-family: Arial, sans-serif;
            font-weight: bold;
          }

          #${contentId} {
            width: 100%;
            margin: 0;
            padding: 0;
          }

          #${contentId} table {
            width: 100%;
            table-layout: fixed;
            border-collapse: collapse;
            margin: 0;
            padding: 0;
          }

          #${contentId} td,
          #${contentId} th,
          #${contentId} p,
          #${contentId} span,
          #${contentId} div,
          #${contentId} h1,
          #${contentId} h2,
          #${contentId} h3,
          #${contentId} h4,
          #${contentId} h5,
          #${contentId} h6 {
            padding: 2mm 1mm;
            font-size: 22px;
            line-height: 1.5;
            font-weight: bold;
            white-space: normal;
            word-wrap: break-word;
            color: rgb(0, 0, 0) !important;
            -webkit-text-fill-color: rgb(0, 0, 0) !important;
          }

          #${contentId} h1 {
            font-size: 28px !important;
          }
          #${contentId} h2 {
            font-size: 26px !important;
          }
          #${contentId} h3 {
            font-size: 24px !important;
          }

          #${contentId} img {
            width: 100%;
            max-width: none;
            height: auto;
            margin: 0 auto;
            display: block;
          }

          @media print {
            * {
              color: rgb(0, 0, 0) !important;
              -webkit-text-fill-color: rgb(0, 0, 0) !important;
              -webkit-print-color-adjust: exact !important;
              print-color-adjust: exact !important;
            }
            body {
              width: 80%;
            }
          }
        </style>
      </head>
      <body>
        ${printableContent.innerHTML}
        <script>
          window.onload = function() {
            window.print();
            setTimeout(function() {
              window.close();
            }, 500);
          }
        </script>
      </body>
    </html>
  `.trim();

  printWindow.document.write(htmlContent);
  printWindow.document.close();
};
