<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Models\Sucursal;

class ConfigImpresion extends Model
{
    use HasFactory;

    protected $table = 'config_impresiones';

    protected $fillable = [
        'sucursal_id',
        'nombre',
        'tipo_impresion',
        'ancho_papel',
        'alto_papel',
        'margen_superior',
        'margen_inferior',
        'margen_izquierdo',
        'margen_derecho',
        'fuente_principal',
        'tamano_fuente_principal',
        'tamano_fuente_titulo',
        'tamano_fuente_subtitulo',
        'tamano_fuente_detalle',
        'color_texto',
        'peso_fuente',
        'interlineado',
        'alineacion_titulo',
        'alineacion_texto',
        'alineacion_empresa',
        'alineacion_direccion',
        'alineacion_folio_fecha',
        'alineacion_total',
        'alineacion_footer',
        'mostrar_logo',
        'mostrar_fecha',
        'mostrar_hora',
        'mostrar_cliente',
        'mostrar_empleado',
        'mostrar_observaciones',
        'separador_lineas',
        'padding_celda',
        'ancho_tabla',
        'ancho_contenido_porcentaje',
        'ancho_cuerpo_porcentaje',
        'es_activa',
        'es_predeterminada',
        'es_impresion_directa',
        'ip_impresora',
        'ngrok_key',
        'user_id'
    ];

    protected $casts = [
        'mostrar_logo' => 'boolean',
        'mostrar_fecha' => 'boolean',
        'mostrar_hora' => 'boolean',
        'mostrar_cliente' => 'boolean',
        'mostrar_empleado' => 'boolean',
        'mostrar_observaciones' => 'boolean',
        'es_activa' => 'boolean',
        'es_predeterminada' => 'boolean',
        'es_impresion_directa' => 'boolean',
        'interlineado' => 'decimal:1',
        'ancho_papel' => 'integer',
        'alto_papel' => 'integer',
        'margen_superior' => 'decimal:1',
        'margen_inferior' => 'decimal:1',
        'margen_izquierdo' => 'decimal:1',
        'margen_derecho' => 'decimal:1',
        'tamano_fuente_principal' => 'integer',
        'tamano_fuente_titulo' => 'integer',
        'tamano_fuente_subtitulo' => 'integer',
        'tamano_fuente_detalle' => 'integer',
        'padding_celda' => 'decimal:1',
        'ancho_tabla' => 'integer',
        'ancho_contenido_porcentaje' => 'integer',
        'ancho_cuerpo_porcentaje' => 'integer',
        'user_id' => 'integer'
    ];

    /**
     * Relación con la sucursal (mantener para compatibilidad)
     */
    public function sucursal(): BelongsTo
    {
        return $this->belongsTo(Sucursal::class);
    }

    /**
     * Relación many-to-many con sucursales
     */
    public function sucursales(): BelongsToMany
    {
        return $this->belongsToMany(Sucursal::class, 'config_impresion_sucursal', 'config_impresion_id', 'sucursal_id')
                    ->withTimestamps();
    }

    /**
     * Relación con el usuario (mantener para compatibilidad)
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Relación many-to-many con usuarios
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'config_impresion_user', 'config_impresion_id', 'user_id')
                    ->withTimestamps();
    }

    /**
     * Obtener la configuración activa por tipo de impresión y sucursal
     */
    public static function getConfigActiva($tipoImpresion = 'ticket', $sucursalId = null)
    {
        $query = self::where('tipo_impresion', $tipoImpresion)
                     ->where('es_activa', true);
        
        if ($sucursalId) {
            $query->where('sucursal_id', $sucursalId);
        }
        
        return $query->first();
    }

    /**
     * Obtener la configuración predeterminada por tipo de impresión y sucursal
     */
    public static function getConfigPredeterminada($tipoImpresion = 'ticket', $sucursalId = null)
    {
        $query = self::where('tipo_impresion', $tipoImpresion)
                     ->where('es_predeterminada', true);
        
        if ($sucursalId) {
            $query->where('sucursal_id', $sucursalId);
        }
        
        return $query->first();
    }

    /**
     * Obtener configuración de impresión directa por usuario, sucursal y tipo
     * Busca en las relaciones many-to-many de usuarios y sucursales
     */
    public static function getConfigDirecta($tipoImpresion, $userId, $sucursalId)
    {
        return self::where('tipo_impresion', $tipoImpresion)
            ->where('es_impresion_directa', true)
            ->where('es_activa', true)
            ->whereHas('users', function($query) use ($userId) {
                $query->where('users.id', $userId);
            })
            ->whereHas('sucursales', function($query) use ($sucursalId) {
                $query->where('sucursales.id', $sucursalId);
            })
            ->with(['sucursales', 'users'])
            ->first();
    }

    /**
     * Activar una configuración
     * Solo inactiva otras configuraciones cuando:
     * - Es predeterminada
     * - No es impresión directa
     * - Del mismo tipo de impresión
     * Para evitar que existan 2 impresoras del mismo tipo que no sean directas y predeterminadas
     */
    public function activar()
    {
        DB::transaction(function () {
            // Solo inactivar otras configuraciones si esta es predeterminada y no es directa
            if ($this->es_predeterminada && !$this->es_impresion_directa) {
                $query = self::where('tipo_impresion', $this->tipo_impresion)
                             ->where('id', '<>', $this->id)
                             ->where('es_impresion_directa', false)
                             ->where('es_predeterminada', true);
                
                $query->update(['es_activa' => false]);
            }

            // Activar esta configuración
            $this->forceFill(['es_activa' => true])->save();
        });
    }

    /**
     * Establecer como predeterminada (quitar predeterminada a las demás del mismo tipo)
     * Si no es impresión directa, también inactiva otras predeterminadas del mismo tipo
     */
    public function establecerPredeterminada()
    {
        DB::transaction(function () {
            // Quitar predeterminada a todas las demás del mismo tipo
            self::where('tipo_impresion', $this->tipo_impresion)
                ->where('id', '<>', $this->id)
                ->update(['es_predeterminada' => false]);
            
            // Si no es impresión directa, inactivar otras predeterminadas del mismo tipo
            if (!$this->es_impresion_directa) {
                self::where('tipo_impresion', $this->tipo_impresion)
                    ->where('id', '<>', $this->id)
                    ->where('es_impresion_directa', false)
                    ->where('es_predeterminada', true)
                    ->update(['es_activa' => false]);
            }
            
            // Establecer esta como predeterminada
            $this->update(['es_predeterminada' => true]);
        });
    }

    /**
     * Obtener estilos CSS para la impresión
     */
    public function getEstilosCSS()
    {
        $anchoPapel = $this->ancho_papel . 'mm';
        $altoPapel = $this->alto_papel > 0 ? $this->alto_papel . 'mm' : 'auto';
        
        return "
            @page {
                size: {$anchoPapel} {$altoPapel};
                margin: {$this->margen_superior}mm {$this->margen_derecho}mm {$this->margen_inferior}mm {$this->margen_izquierdo}mm;
            }

            * {
                color: {$this->color_texto} !important;
                -webkit-text-fill-color: {$this->color_texto} !important;
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
            }

            body {
                margin: 0;
                padding: 0;
                width: {$this->ancho_cuerpo_porcentaje}%;
                font-family: {$this->fuente_principal}, sans-serif;
                font-weight: {$this->peso_fuente};
                line-height: {$this->interlineado};
            }

            .ticket-content {
                width: {$this->ancho_contenido_porcentaje}%;
                margin: 0;
                padding: 0;
            }

            .ticket-content table {
                width: {$this->ancho_tabla}%;
                table-layout: fixed;
                border-collapse: collapse;
                margin: 0;
                padding: 0;
            }

            .ticket-content td,
            .ticket-content th,
            .ticket-content p,
            .ticket-content span,
            .ticket-content div {
                padding: {$this->padding_celda}mm 1mm;
                font-size: {$this->tamano_fuente_principal}px;
                line-height: {$this->interlineado};
                font-weight: {$this->peso_fuente};
                white-space: normal;
                word-wrap: break-word;
                color: {$this->color_texto} !important;
                -webkit-text-fill-color: {$this->color_texto} !important;
                text-align: {$this->alineacion_texto};
            }

            .ticket-content h1 {
                font-size: {$this->tamano_fuente_titulo}px !important;
                text-align: {$this->alineacion_titulo};
            }
            
            .ticket-content h2 {
                font-size: {$this->tamano_fuente_subtitulo}px !important;
                text-align: {$this->alineacion_titulo};
            }
            
            .ticket-content h3 {
                font-size: {$this->tamano_fuente_subtitulo}px !important;
                text-align: {$this->alineacion_titulo};
            }

            .ticket-content .detalle {
                font-size: {$this->tamano_fuente_detalle}px !important;
            }

            /* Estilos específicos para elementos del ticket */
            .ticket-content .empresa-nombre {
                font-size: {$this->tamano_fuente_titulo}px !important;
                font-weight: bold !important;
                text-align: {$this->alineacion_empresa} !important;
                margin: 0;
                padding: 2mm 0;
            }

            .ticket-content .empresa-tagline {
                font-size: {$this->tamano_fuente_principal}px !important;
                text-align: {$this->alineacion_empresa} !important;
                margin: 0;
                padding: 1mm 0;
            }

            .ticket-content .empresa-direccion {
                font-size: {$this->tamano_fuente_detalle}px !important;
                text-align: {$this->alineacion_direccion} !important;
                margin: 0;
                padding: 1mm 0;
            }

            .ticket-content .folio-fecha {
                font-size: {$this->tamano_fuente_detalle}px !important;
                text-align: {$this->alineacion_folio_fecha} !important;
                margin: 0;
                padding: 1mm 0;
            }

            .ticket-content .total {
                font-size: {$this->tamano_fuente_principal}px !important;
                font-weight: bold !important;
                text-align: {$this->alineacion_total} !important;
                margin: 0;
                padding: 2mm 0;
            }

            .ticket-content .footer {
                font-size: {$this->tamano_fuente_detalle}px !important;
                font-weight: bold !important;
                text-align: {$this->alineacion_footer} !important;
                margin: 0;
                padding: 1mm 0;
            }

            .ticket-content img {
                width: 100%;
                max-width: none;
                height: auto;
                margin: 0 auto;
                display: block;
            }

            .separador {
                text-align: center;
                font-size: {$this->tamano_fuente_detalle}px;
                margin: 2mm 0;
            }

            @media print {
                * {
                    color: {$this->color_texto} !important;
                    -webkit-text-fill-color: {$this->color_texto} !important;
                    -webkit-print-color-adjust: exact !important;
                    print-color-adjust: exact !important;
                }
                body {
                    width: {$this->ancho_cuerpo_porcentaje}%;
                }
            }
        ";
    }
}
