<?php

namespace App\Http\Controllers;

use App\Models\Pago;
use App\Models\Sucursal;
use App\Models\Tienda;
use App\Models\TipoPago;
use App\Models\Venta;
use App\Models\VentaDetalle;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Mail;
use App\Mail\AvisoNuevoVentaAdmin;
use App\Mail\AvisoNuevoVentaCliente;
use App\Helpers\CorreoHelper;


class TiendaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): JsonResponse
    {
        $tienda=Tienda::leftjoin('productos', 'tiendas.producto_id', '=', 'productos.id')
        ->selectRaw('
            tiendas.*,
            productos.nombre nombre_producto,
            (productos.precio) precio_unitario,
            tiendas.cantidad*(productos.precio) total
        ')
        ->where(['user_id'=>$request->user()->id])->get();

         $sumTiendaCliente=Tienda::leftjoin('productos', 'tiendas.producto_id', '=', 'productos.id')
        ->selectRaw('
            SUM(tiendas.cantidad*(productos.precio)) sum
        ')
        ->where(['user_id'=>$request->user()->id])
        ->first();

        return response()->json([
            'data' =>$tienda->load('producto'),
            'datasum' =>$sumTiendaCliente
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'producto_id' => ['required', Rule::exists('productos', 'id')],
            'user_id' => ['required', Rule::exists('users', 'id')],
            'cantidad' => ['required'],
        ]);

        $cantidad = 0;
        $buscarTienda = Tienda::where(['producto_id'=>$request->producto_id, 'user_id'=>$request->user_id])->first();
        if($buscarTienda)
        {
            $cantidad = $buscarTienda->cantidad +  $request->cantidad;
            $tienda = $buscarTienda->update(['cantidad'=>$cantidad]);
        } else {
            $params= $request->only([
                'producto_id',
                'user_id',
                'cantidad'
            ]);

            $tienda = Tienda::create($params);
        }


        return response()->json([
            'data' => $tienda,
        ], JsonResponse::HTTP_OK);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Tienda $tienda): JsonResponse
    {
        return response()->json([
            'data' => [
                'tienda'=>$tienda->load('producto')->load('user')
            ],
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Tienda $tienda): JsonResponse
    {
        $request->validate([
            'producto_id' => ['required', Rule::exists('productos', 'id')],
            'user_id' => ['required', Rule::exists('users', 'id')],
            'cantidad' => ['required'],
        ]);


        $params= $request->only([
            'producto_id',
            'user_id',
            'cantidad'
        ]);

        $tienda->update($params);

        return response()->json([
            'data' => $tienda,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Tienda $tienda): JsonResponse
    {
        $tienda->delete();
        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroyByUser(Request $request): JsonResponse
    {
        $tiendaCliente = Tienda::where(['user_id'=>$request->user()->id])->get();
        foreach ($tiendaCliente as $key => $value) {
            $value->delete();
        }
        return response()->json(null, JsonResponse::HTTP_OK);
    }

    public function count(Request $request): JsonResponse
    {
        $conteoTiendaCliente=Tienda::where(['user_id'=>$request->user()->id])
        ->selectRaw('
            count(producto_id) count
        ')->first();
        return response()->json([
            'data' =>$conteoTiendaCliente,
        ], JsonResponse::HTTP_OK);
    }

    public function pagar(Request $request): JsonResponse
    {
        $request->validate([
            'direccion_id' => ['required', Rule::exists('direcciones_entrega', 'id')],
            'datos_fiscales_id' => ['nullable', Rule::exists('datos_fiscales', 'id')],
            'opcion_pago_transferencia_id' => ['required', Rule::exists('tipos_pagos', 'id')],
            'comprobante_pago' => ['nullable', 'file', 'mimes:pdf,jpg,jpeg,png', 'max:2048'],
        ]);

        $path = null;
        if($request->comprobante_pago){
            $path = $request->comprobante_pago->store('public/images/productos');
        }

        $sucursal = Sucursal::where(['estatus'=>'activo'])->first();
        $tipo_pago = TipoPago::where(['descripcion'=>'Transferencias'])->first();

        $tienda = Tienda::where(['user_id'=>$request->user()->id])->get();

        $total = 0;

        foreach ($tienda as $value) {
            $total += $value->cantidad * $value->producto->precio;
        }

        $params = [
            'sucursal_id' => $sucursal->id,
            'cliente_id' => $request->cliente_id,
            'total' => $total,
            'subtotal' => $total,
            'iva' => 0,
            'es_pago_tienda' => 'no',
            'es_tienda_online' => 'si',
            'direccion_id' => $request->direccion_id,
            'opcion_pago_transferencia_id' => $request->opcion_pago_transferencia_id,
            'comprobante_pago' => $path,
        ];

        if ($request->datos_fiscales_id) {
            $params['datos_fiscales_id'] = $request->datos_fiscales_id;
        }

        $venta = Venta::create($params);
        usleep(500000);
        Pago ::create([
            'venta_id' => $venta->id,
            'tipo_pago_id' => $tipo_pago->id,
            'pago' => $total,
            'banco' => null,
            'folio_pago' => null,
            'cambio' => 0,
            'total' => $total,
        ]);
        foreach ($tienda as $value) {
            $params_det =[
                'descripcion'=>$value->producto->nombre,
                'venta_id'=>$venta->id,
                'producto_id'=>$value->producto_id,
                'tratamiento_id'=>null,
                'vale_id'=>null,
                'total'=>$value->cantidad * $value->producto->precio,
                'subtotal'=>$value->cantidad * $value->producto->precio,
                'precio_unitario'=>$value->producto->precio,
                'cantidad'=>$value->cantidad,
                'iva'=>$value->producto->iva,
                'ieps'=>$value->producto->ieps,
                'es_pago_tienda'=>'no',
            ];
            VentaDetalle::create($params_det);
            usleep(500000);
        }

        foreach ($tienda as $value) {
            $value->delete();
        }
        $venta = Venta::find($venta->id);

        if($path){
            // enviar correo usando la nueva configuración
            $correo_admin = CorreoHelper::getCorreoDestino(env('MAIL_USERNAME'));
            $correo_cliente = CorreoHelper::getCorreoDestino($venta->cliente->email ?? null);

            Mail::to($correo_admin)->send(new AvisoNuevoVentaAdmin($venta));
            Mail::to($correo_cliente)->send(new AvisoNuevoVentaCliente($venta));
        }

        return response()->json([
            'data' => [],
        ], JsonResponse::HTTP_OK);

    }

    public function updateComprobante(Request $request, Venta $venta): JsonResponse
    {
        $request->validate([
            'comprobante_pago' => ['required', 'file', 'mimes:pdf,jpg,jpeg,png', 'max:2048'],
        ]);

        $path = null;
        if($request->comprobante_pago){
            $path = $request->comprobante_pago->store('public/images/productos');
        }

        if($path){
            $venta = Venta::find($venta->id);
            // enviar correo usando la nueva configuración
            $correo_admin = CorreoHelper::getCorreoDestino(env('MAIL_USERNAME'));
            $correo_cliente = CorreoHelper::getCorreoDestino($venta->cliente->email ?? null);

            Mail::to($correo_admin)->send(new AvisoNuevoVentaAdmin($venta));
            Mail::to($correo_cliente)->send(new AvisoNuevoVentaCliente($venta));
        }

        $venta->update(['comprobante_pago' => $path]);
        return response()->json([
            'data' => $venta,
        ], JsonResponse::HTTP_OK);
    }
}
