<?php

namespace App\Http\Controllers;

use App\Models\ProductoBaja;
use App\Models\Inventario;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Database\Eloquent\Builder;

class ProductoBajaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(): JsonResponse
    {
        $salida=ProductoBaja::where(['estatus'=>'activo'])->get();
        return response()->json([
            'data' =>$salida,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'producto_id' => ['required', Rule::exists('productos', 'id')],
            'tipo_baja_id' => ['required', Rule::exists('tipos_bajas', 'id')],
            'cantidad' => ['required', 'numeric', 'max:999999999.99'],
            'observacion' => ['required'],
        ]);

        $salida = ProductoBaja::create($request->only([
            'sucursal_id',
            'producto_id',
            'tipo_baja_id',
            'cantidad',
            'observacion',
        ]));

        $model_inventario=Inventario::where($request->only([
            'sucursal_id',
            'producto_id'
        ]))->first();

        if($model_inventario){
            $cantidad=$model_inventario->cantidad_existente-$request->cantidad;
            $model_inventario->update([
                'cantidad_existente' => $cantidad
            ]);
        } else {
            $salida = Inventario::create([
                'sucursal_id'=>$request->sucursal_id,
                'producto_id'=>$request->producto_id,
                'cantidad_existente'=> (-$request->cantidad),
                'cantidad_vendido'=>0,
            ]);
        }

        return response()->json([
            'data' => $salida,
        ], JsonResponse::HTTP_OK);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(ProductoBaja $salida): JsonResponse
    {
        return response()->json([
            'data' => $salida,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, ProductoBaja $salida): JsonResponse
    {
        $request->validate([
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'producto_id' => ['required', Rule::exists('productos', 'id')],
            'tipo_baja_id' => ['required', Rule::exists('tipos_bajas', 'id')],
            'cantidad' => ['required', 'numeric', 'max:999999999.99'],
            'observacion' => ['required'],
        ]);

        $salida->update($request->only([
            'sucursal_id',
            'producto_id',
            'tipo_baja_id',
            'cantidad',
            'observacion',
        ]));

        return response()->json([
            'data' => $salida,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(ProductoBaja $salida): JsonResponse
    {
        $salida->delete();
        // return $salida;

        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }

    public function paginacion(Request $request): JsonResponse
    {
        $producto=ProductoBaja::leftjoin('productos', 'productos.id', '=', 'productos_bajas.producto_id')
        ->leftjoin('sucursales', 'sucursales.id', '=', 'productos_bajas.sucursal_id')
        ->leftjoin('tipos_bajas', 'tipos_bajas.id', '=', 'productos_bajas.tipo_baja_id')
        ->selectRaw("
            productos_bajas.*,
            date_format(productos_bajas.fecha_registro,'%d-%m-%Y %l %p') fecha_hora,
            tipos_bajas.descripcion tipo_baja,
            productos.nombre producto,
            productos.descripcion,
            productos.precio,
            productos.path,
            (productos.precio*(iva/100)) iva,
            (productos.precio + (productos.precio*(iva/100))) total,
            sucursales.nombre sucursal
        ")
        ->when($request->search,function (Builder $builder) use ($request) {
            $builder->orWhere('productos.id', 'like', "%{$request->search}%")
                ->orWhere('productos.nombre', 'like', "%{$request->search}%")
                ->orWhere('productos.descripcion', 'like', "%{$request->search}%")
                ->orWhere('productos.precio', 'like', "%{$request->search}%")
                ->orWhere('sucursales.nombre', 'like', "%{$request->search}%")
                ->orWhere('productos_bajas.cantidad', 'like', "%{$request->search}%")
                ->orWhereRaw("date_format(productos_bajas.fecha_registro,'%d-%m-%Y %l %p') LIKE '%{$request->search}%'");
        })
        ->when($request->sucursal_id,function (Builder $builder) use ($request) {
            $builder->Where(['sucursales.id'=>$request->sucursal_id]);
        })
        ->where(['productos.estatus'=>'activo'])->orderBy('updated_at','ASC')->paginate($request->perPage);
        return response()->json([
            'data' =>$producto,
        ], JsonResponse::HTTP_OK);
    }
}
