<?php

namespace App\Http\Controllers;

use App\Models\OpcionPagoTransferencia;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class OpcionPagoTransferenciaController extends Controller
{
    // Listar todas las opciones de pago por transferencia
    public function index(): JsonResponse
    {
        $opciones = OpcionPagoTransferencia::get();
        return response()->json(['data' => $opciones]);
    }

    // Mostrar una opción específica
    public function show($id): JsonResponse
    {
        $opcion = OpcionPagoTransferencia::findOrFail($id);
        return response()->json(['data' => $opcion]);
    }

    // Crear una nueva opción
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'banco'   => 'required|string|max:255',
            'titular' => 'required|string|max:500',
            'clabe'   => 'required|string|max:18',
            'cuenta'  => 'required|string|max:20',
            'estatus' => 'in:activo,inactivo',
            'es_principal' => 'in:si,no'
        ]);

        $opcion = OpcionPagoTransferencia::create($validated);

        return response()->json(['data' => $opcion, 'message' => 'Opción de pago creada correctamente.'], 201);
    }

    // Actualizar una opción existente
    public function update(Request $request, OpcionPagoTransferencia $opcion): JsonResponse
    {

        $request->validate([
            'banco'   => 'sometimes|required|string|max:255',
            'titular' => 'sometimes|required|string|max:500',
            'clabe'   => 'sometimes|required|string|max:18',
            'cuenta'  => 'sometimes|required|string|max:20',
            'estatus' => 'in:activo,inactivo',
            'es_principal' => 'in:si,no'
        ]);

        $opcion->update($request->only(['banco', 'titular', 'clabe', 'cuenta', 'estatus']));

        return response()->json(['data' => $opcion, 'message' => 'Opción de pago actualizada correctamente.']);
    }

    // Eliminar una opción
    public function destroy($id): JsonResponse
    {
        $opcion = OpcionPagoTransferencia::findOrFail($id);
        $opcion->delete();

        return response()->json(['message' => 'Opción de pago eliminada correctamente.']);
    }

    public function getForSelect(): JsonResponse
    {
        $opciones = OpcionPagoTransferencia::where('estatus', 'activo')->get();
        return response()->json(['data' => $opciones]);
    }

    public function principal(): JsonResponse
    {
        $opciones = OpcionPagoTransferencia::where('estatus', 'activo')->where('es_principal', 'si')->first();
        return response()->json(['data' => $opciones]);
    }
}
