<?php

namespace App\Http\Controllers;

use App\Models\Inventario;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Database\Eloquent\Builder;

class InventarioController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(): JsonResponse
    {
        $inventario=Inventario::where(['estatus'=>'activo'])->get();
        return response()->json([
            'data' =>$inventario,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'nombre' => ['required', 'max:500'],
        ]);

        $inventario = Inventario::create($request->only([
            'sucursal_id',
            'nombre',
        ]));

        return response()->json([
            'data' => $inventario,
        ], JsonResponse::HTTP_OK);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Inventario $inventario): JsonResponse
    {
        return response()->json([
            'data' => $inventario,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Inventario $inventario): JsonResponse
    {
        $request->validate([
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'nombre' => ['required', 'max:500'],
        ]);

        $inventario->update($request->only([
            'sucursal_id',
            'nombre',
        ]));

        return response()->json([
            'data' => $inventario,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Inventario $inventario): JsonResponse
    {
        if (
            \count($inventario->inventarios) > 0 ||
            \count($inventario->ingresosMercancias) > 0 ||
            \count($inventario->productosBajas) > 0 
        ) {
            $inventario->update([
                'estatus' => 'inactivo',
            ]);
        } else {
            $inventario->delete();
        }

        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }

    public function paginacion(Request $request): JsonResponse
    {
        $producto=Inventario::leftjoin('productos', 'productos.id', '=', 'inventarios.producto_id')
        ->leftjoin('sucursales', 'sucursales.id', '=', 'inventarios.sucursal_id')
        ->selectRaw('
            inventarios.*,
            productos.nombre,
            productos.descripcion,
            productos.precio,
            productos.path,
            (productos.precio*(iva/100)) iva,
            (productos.precio + (productos.precio*(iva/100))) total,
            sucursales.nombre sucursal
        ')
        ->when($request->search,function (Builder $builder) use ($request) {
            $builder->orWhere('productos.id', 'like', "%{$request->search}%")
                ->orWhere('productos.nombre', 'like', "%{$request->search}%")
                ->orWhere('productos.descripcion', 'like', "%{$request->search}%")
                ->orWhere('productos.precio', 'like', "%{$request->search}%")
                ->orWhere('sucursales.nombre', 'like', "%{$request->search}%")
                ->orWhere('inventarios.cantidad_existente', 'like', "%{$request->search}%")
                ->orWhere('inventarios.cantidad_vendido', 'like', "%{$request->search}%");
        })
        ->when($request->sucursal_id,function (Builder $builder) use ($request) {
            $builder->Where(['sucursales.id'=>$request->sucursal_id]);
        })
        ->where(['productos.estatus'=>'activo'])->orderBy('updated_at','ASC')->paginate($request->perPage);
        return response()->json([
            'data' =>$producto,
        ], JsonResponse::HTTP_OK);
    }
}
