<?php

namespace App\Http\Controllers;

use App\Models\IngresoMercancia;
use App\Models\Inventario;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Database\Eloquent\Builder;

class IngresoMercanciaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(): JsonResponse
    {
        $ingreso=IngresoMercancia::where(['estatus'=>'activo'])->get();
        return response()->json([
            'data' =>$ingreso,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'producto_id' => ['required', Rule::exists('productos', 'id')],
            'cantidad' => ['required', 'numeric', 'max:999999999.99'],
            'costo' => ['required', 'numeric', 'max:999999999.99'],
            'observacion' => ['required'],
        ]);

        $ingreso = IngresoMercancia::create($request->only([
            'sucursal_id',
            'producto_id',
            'cantidad',
            'costo',
            'observacion',
        ]));

        $model_inventario=Inventario::where($request->only([
            'sucursal_id',
            'producto_id'
        ]))->first();

        if($model_inventario){
            $cantidad=$model_inventario->cantidad_existente+$request->cantidad;
            $model_inventario->update([
                'cantidad_existente' => $cantidad
            ]);
        } else {
            $ingreso = Inventario::create([
                'sucursal_id'=>$request->sucursal_id,
                'producto_id'=>$request->producto_id,
                'cantidad_existente'=>$request->cantidad,
                'cantidad_vendido'=>0,
            ]);
        }

        return response()->json([
            'data' => $ingreso,
        ], JsonResponse::HTTP_OK);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(IngresoMercancia $ingreso): JsonResponse
    {
        return response()->json([
            'data' => $ingreso,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, IngresoMercancia $ingreso): JsonResponse
    {
        $request->validate([
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'producto_id' => ['required', Rule::exists('productos', 'id')],
            'cantidad' => ['required', 'numeric', 'max:999999999.99'],
            'costo' => ['required', 'numeric', 'max:999999999.99'],
            'observacion' => ['required'],
        ]);

        $ingreso->update($request->only([
            'sucursal_id',
            'producto_id',
            'cantidad',
            'costo',
            'observacion',
        ]));

        return response()->json([
            'data' => $ingreso,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(IngresoMercancia $ingreso): JsonResponse
    {
        $ingreso->delete();
        // return $ingreso;

        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }

    public function paginacion(Request $request): JsonResponse
    {
        $producto=IngresoMercancia::leftjoin('productos', 'productos.id', '=', 'ingresos_mercancias.producto_id')
        ->leftjoin('sucursales', 'sucursales.id', '=', 'ingresos_mercancias.sucursal_id')
        ->selectRaw("
            ingresos_mercancias.*,
            date_format(ingresos_mercancias.fecha_registro,'%d-%m-%Y %l %p') fecha_hora,
            productos.nombre producto,
            productos.descripcion,
            productos.precio,
            productos.path,
            (productos.precio*(iva/100)) iva,
            (productos.precio + (productos.precio*(iva/100))) total,
            sucursales.nombre sucursal
        ")
        ->when($request->search,function (Builder $builder) use ($request) {
            $builder->orWhere('productos.id', 'like', "%{$request->search}%")
                ->orWhere('productos.nombre', 'like', "%{$request->search}%")
                ->orWhere('productos.descripcion', 'like', "%{$request->search}%")
                ->orWhere('productos.precio', 'like', "%{$request->search}%")
                ->orWhere('sucursales.nombre', 'like', "%{$request->search}%")
                ->orWhere('ingresos_mercancias.cantidad', 'like', "%{$request->search}%")
                ->orWhereRaw("date_format(ingresos_mercancias.fecha_registro,'%d-%m-%Y %l %p') LIKE '%{$request->search}%'");
        })
        ->when($request->sucursal_id,function (Builder $builder) use ($request) {
            $builder->Where(['sucursales.id'=>$request->sucursal_id]);
        })
        ->where(['productos.estatus'=>'activo'])->orderBy('updated_at','ASC')->paginate($request->perPage);
        return response()->json([
            'data' =>$producto,
        ], JsonResponse::HTTP_OK);
    }
}
