<?php

namespace App\Http\Controllers;

use App\Models\Vale;
use App\Models\VentaDetalle;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ImprimirValeController extends Controller
{
    /**
     * Imprimir vale
     * Si IMPRIMIR_DIRECTO=true, envía la petición al endpoint externo
     * Si no, retorna el contenido formateado para impresión local
     */
    public function imprimir(Request $request, $valeId): JsonResponse
    {
        try {
            // Buscar el vale con sus relaciones
            $vale = Vale::find($valeId);

            if (!$vale) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vale no encontrado'
                ], 404);
            }

            // Buscar el detalle de venta relacionado para obtener el tratamiento
            $ventaDetalle = VentaDetalle::where('vale_id', $vale->id)
                ->with('tratamiento')
                ->first();

            // Construir el contenido del vale
            $datosVale = $this->construirDatosVale($vale, $ventaDetalle);

            // Verificar si se debe imprimir directamente
            $imprimirDirecto = env('IMPRIMIR_DIRECTO', false) === true || env('IMPRIMIR_DIRECTO', 'false') === 'true';

            if ($imprimirDirecto) {
                // Enviar al endpoint externo
                return $this->enviarAEndpointExterno($datosVale, $request);
            }

            // Retornar contenido para impresión local
            return response()->json([
                'success' => true,
                'data' => [
                    'vale' => $datosVale,
                    'vale_id' => $vale->id
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error al imprimir vale: ' . $e->getMessage(), [
                'vale_id' => $valeId ?? null,
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error al procesar la impresión del vale',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Construir los datos del vale desde el modelo
     */
    private function construirDatosVale(Vale $vale, $ventaDetalle): array
    {
        $datos = [
            'folio' => (string)$vale->id,
            'vigencia' => $vale->vigencia 
                ? date('d-m-Y', strtotime($vale->vigencia))
                : date('d-m-Y', strtotime('+31 days')),
            'tratamiento' => 'Vale de Tratamiento',
            'token' => $vale->token ?? null
        ];

        // Obtener el nombre del tratamiento si existe
        if ($ventaDetalle && $ventaDetalle->tratamiento) {
            $datos['tratamiento'] = $ventaDetalle->tratamiento->nombre ?? $ventaDetalle->tratamiento->tratamiento ?? 'Vale de Tratamiento';
        } elseif ($ventaDetalle && $ventaDetalle->descripcion) {
            // Si no hay tratamiento, usar la descripción del detalle
            $datos['tratamiento'] = $ventaDetalle->descripcion;
        }

        return $datos;
    }

    /**
     * Enviar el vale al endpoint externo de impresión
     */
    private function enviarAEndpointExterno(array $datosVale, Request $request): JsonResponse
    {
        $endpointUrl = env('IMPRESION_ENDPOINT_URL', 'http://localhost:8081/api');
        $url = rtrim($endpointUrl, '/') . '/imprimir-vale';

        // Obtener parámetros de la petición o usar valores por defecto
        $nombreImpresora = $request->input('nombreImpresora', 'ticket');
        $anchoPapel = $request->input('anchoPapel', 600);
        $incluirImagen = $request->input('incluirImagen', true);

        $payload = [
            'nombreImpresora' => $nombreImpresora,
            'folio' => $datosVale['folio'],
            'vigencia' => $datosVale['vigencia'],
            'tratamiento' => $datosVale['tratamiento'],
            'anchoPapel' => (int)$anchoPapel,
            'incluirImagen' => (bool)$incluirImagen
        ];

        try {
            $response = Http::timeout(30)->post($url, $payload);

            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => $response->json()['message'] ?? 'Vale enviado correctamente',
                    'data' => $response->json()
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Error al enviar el vale al servicio de impresión',
                'error' => $response->body(),
                'status_code' => $response->status()
            ], $response->status());

        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('Error de conexión al servicio de impresión: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'No se pudo conectar al servicio de impresión',
                'error' => $e->getMessage()
            ], 503);

        } catch (\Exception $e) {
            Log::error('Error al enviar vale al endpoint externo: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error al enviar el vale',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}

