<?php

namespace App\Http\Controllers;

use App\Models\Envio;
use App\Models\Venta;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class EnvioController extends Controller
{
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'venta_id' => ['required', Rule::exists('ventas', 'id')],
            'paqueteria' => ['required', 'string', 'max:255'],
            'numero_guia' => ['required', 'string', 'max:255'],
            'fecha_envio' => ['required', 'date'],
            'fecha_entrega_estimada' => ['required', 'date', 'after:fecha_envio'],
        ], [
            'venta_id.required' => 'El ID de venta es obligatorio',
            'venta_id.exists' => 'La venta especificada no existe',
            'paqueteria.required' => 'La paquetería es obligatoria',
            'numero_guia.required' => 'El número de guía es obligatorio',
            'fecha_envio.required' => 'La fecha de envío es obligatoria',
            'fecha_envio.date' => 'La fecha de envío debe ser una fecha válida',
            'fecha_entrega_estimada.required' => 'La fecha de entrega estimada es obligatoria',
            'fecha_entrega_estimada.date' => 'La fecha de entrega estimada debe ser una fecha válida',
            'fecha_entrega_estimada.after' => 'La fecha de entrega estimada debe ser posterior a la fecha de envío',
        ]);

        // Verificar si ya existe un envío para esta venta
        $envioExistente = Envio::where('venta_id', $request->venta_id)->first();
        if ($envioExistente) {
            return response()->json([
                'error' => 'Ya existe un envío para esta venta',
                'data' => $envioExistente
            ], JsonResponse::HTTP_CONFLICT);
        }

        $envio = Envio::create($request->all());

        // Cargar la relación con la venta
        $envio->load('venta');

        return response()->json([
            'message' => 'Envío creado exitosamente',
            'data' => $envio
        ], JsonResponse::HTTP_CREATED);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Envio $envio): JsonResponse
    {

        $request->validate([
            'paqueteria' => ['sometimes', 'string', 'max:255'],
            'numero_guia' => ['sometimes', 'string', 'max:255'],
            'fecha_envio' => ['sometimes', 'date'],
            'fecha_entrega_estimada' => ['sometimes', 'date', 'after:fecha_envio'],
        ], [
            'paqueteria.string' => 'La paquetería debe ser texto',
            'numero_guia.string' => 'El número de guía debe ser texto',
            'fecha_envio.date' => 'La fecha de envío debe ser una fecha válida',
            'fecha_entrega_estimada.date' => 'La fecha de entrega estimada debe ser una fecha válida',
            'fecha_entrega_estimada.after' => 'La fecha de entrega estimada debe ser posterior a la fecha de envío',
        ]);

        $envio->update($request->all());

        // Cargar la relación con la venta
        $envio->load('venta');

        return response()->json([
            'message' => 'Envío actualizado exitosamente',
            'data' => $envio
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id): JsonResponse
    {
        $envio = Envio::with('venta')->find($id);
        
        if (!$envio) {
            return response()->json([
                'error' => 'Envío no encontrado'
            ], JsonResponse::HTTP_NOT_FOUND);
        }

        return response()->json([
            'data' => $envio
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): JsonResponse
    {
        $query = Envio::with('venta');

        // Filtro por estatus
        if ($request->has('estatus')) {
            $query->where('estatus', $request->estatus);
        }

        // Filtro por venta
        if ($request->has('venta_id')) {
            $query->where('venta_id', $request->venta_id);
        }

        // Filtro por paquetería
        if ($request->has('paqueteria')) {
            $query->where('paqueteria', 'like', '%' . $request->paqueteria . '%');
        }

        // Búsqueda por número de guía
        if ($request->has('search')) {
            $query->where('numero_guia', 'like', '%' . $request->search . '%');
        }

        $envios = $query->orderBy('fecha_envio', 'desc')->paginate($request->perPage ?? 15);

        return response()->json([
            'data' => $envios
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id): JsonResponse
    {
        $envio = Envio::find($id);
        
        if (!$envio) {
            return response()->json([
                'error' => 'Envío no encontrado'
            ], JsonResponse::HTTP_NOT_FOUND);
        }

        $envio->delete();

        return response()->json([
            'message' => 'Envío eliminado exitosamente'
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Obtener envíos por venta
     *
     * @param  int  $ventaId
     * @return \Illuminate\Http\Response
     */
    public function getByVenta($ventaId): JsonResponse
    {
        $envios = Envio::with('venta')
            ->where('venta_id', $ventaId)
            ->orderBy('fecha_envio', 'desc')
            ->get();

        return response()->json([
            'data' => $envios
        ], JsonResponse::HTTP_OK);
    }
}
