<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\DatoFiscal;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

class DatoFiscalController extends Controller
{
    public function index(Cliente $cliente): JsonResponse
    {

        $datosFiscales = DatoFiscal::where('cliente_id', $cliente->id)
            ->where('estatus', 'activo')
            ->orderBy('es_principal', 'desc')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($datoFiscal) {
                $datoFiscal->documento_fiscal_path = $datoFiscal->documento_fiscal
                    ? asset(str_replace("public", "storage", $datoFiscal->documento_fiscal))
                    : null;
                return $datoFiscal;
            });

        return response()->json(['data' => $datosFiscales]);
    }

    public function store(Request $request): JsonResponse
    {
        $validatedData = $request->validate([
            'cliente_id' => ['required', 'integer', Rule::exists('clientes', 'id')],
            'rfc' => ['string', 'max:13'],
            'razon_social' => ['required', 'string', 'max:255'],
            'correo' => ['nullable', 'email', 'max:255'],
            'uso_cfdi' => ['nullable', 'string', 'max:10'],
            'regimen_fiscal' => ['nullable', 'string', 'max:10'],
            'metodo_pago' => ['nullable', 'string', 'max:10'],
            'documento_fiscal_file' => ['required', 'file', 'mimes:pdf'],
            'es_principal' => ['required', Rule::in(['si', 'no'])],
            'estatus' => ['required', Rule::in(['activo', 'inactivo'])],
        ], [
            'cliente_id.required' => 'El cliente es requerido',
            'cliente_id.exists' => 'El cliente seleccionado no existe',
            'razon_social.required' => 'La razón social es requerida',
            'documento_fiscal_file.file' => 'El archivo debe ser un PDF',
            'es_principal.required' => 'Debe especificar si es dato fiscal principal',
            'es_principal.in' => 'El valor debe ser si o no',
            'estatus.in' => 'El estatus debe ser activo o inactivo',
            'documento_fiscal_file.mimes' => 'El archivo debe ser un PDF',
            'documento_fiscal_file.required' => 'El archivo es requerido',
            'estatus.required' => 'El estatus es requerido',
        ]);

        try {
            DB::beginTransaction();

            if ($validatedData['es_principal'] === 'si') {
                DatoFiscal::where('cliente_id', $validatedData['cliente_id'])
                    ->where('es_principal', 'si')
                    ->update(['es_principal' => 'no']);
            }

            // Manejar el archivo si se proporcionó
            // if ($request->hasFile('documento_fiscal_file')) {
            //     // crear directorio /images/documentos_fiscales y si no existe
            //     // if (!Storage::disk('public')->exists('documentos')) {
            //     //     Storage::disk('public')->makeDirectory('documentos');
            //     // }
            //     // if (!Storage::disk('public')->exists('documentos/documentos_fiscales')) {
            //     //     Storage::disk('public')->makeDirectory('documentos/documentos_fiscales');
            //     // }
            //     $path = $request->documento_fiscal_file->store('public/images/productos');
            //     $validatedData['documento_fiscal'] = $path;
            // }
            if(!empty($request->documento_fiscal_file)){
                $directory = 'public/documentos/documentos_fiscales';
                $fullPath = storage_path('app/' . $directory);

                // Crear directorio recursivamente si no existe usando File facade
                if (!File::exists($fullPath)) {
                    File::makeDirectory($fullPath, 0755, true, true);
                }

                $path = $request->documento_fiscal_file->store($directory);
                $validatedData['documento_fiscal'] = $path;
            }

            $datoFiscal = DatoFiscal::create($validatedData);

            DB::commit();

            return response()->json([
                'data' => $datoFiscal,
                'message' => 'Datos fiscales creados exitosamente.'
            ], JsonResponse::HTTP_CREATED);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Error al crear los datos fiscales.',
                'error' => $e->getMessage()
            ], JsonResponse::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    public function show(DatoFiscal $datoFiscal): JsonResponse
    {
        $documentoFiscalPath = $datoFiscal->documento_fiscal
            ? asset(str_replace("public", "storage", $datoFiscal->documento_fiscal))
            : null;

        return response()->json([
            'data' => $datoFiscal,
            'documento_fiscal_path' => $documentoFiscalPath,
        ]);
    }

    public function update(Request $request, DatoFiscal $datoFiscal): JsonResponse
    {
        $validatedData = $request->validate([
            'cliente_id' => ['required', 'integer', Rule::exists('clientes', 'id')],
            'rfc' => ['string', 'max:13'],
            'razon_social' => ['required', 'string', 'max:255'],
            'correo' => ['nullable', 'email', 'max:255'],
            'uso_cfdi' => ['nullable', 'string', 'max:10'],
            'regimen_fiscal' => ['nullable', 'string', 'max:10'],
            'metodo_pago' => ['nullable', 'string', 'max:10'],
            'documento_fiscal_file' => ['nullable', 'file', 'mimes:pdf'],
            'es_principal' => ['required', Rule::in(['si', 'no'])],
            'estatus' => ['required', Rule::in(['activo', 'inactivo'])],
        ], [
            'cliente_id.required' => 'El cliente es requerido',
            'cliente_id.exists' => 'El cliente seleccionado no existe',
            'razon_social.required' => 'La razón social es requerida',
            'documento_fiscal_file.file' => 'El archivo debe ser un PDF',
            'es_principal.required' => 'Debe especificar si es dato fiscal principal',
            'es_principal.in' => 'El valor debe ser si o no',
            'estatus.in' => 'El estatus debe ser activo o inactivo',
            'documento_fiscal_file.mimes' => 'El archivo debe ser un PDF',
            'estatus.required' => 'El estatus es requerido',
        ]);

        try {
            DB::beginTransaction();

            if (isset($validatedData['es_principal']) && $validatedData['es_principal'] === 'si') {
                DatoFiscal::where('cliente_id', $datoFiscal->cliente_id)
                    ->where('id', '!=', $datoFiscal->id)
                    ->where('es_principal', 'si')
                    ->update(['es_principal' => 'no']);
            }
            if(!empty($request->documento_fiscal_file)){
                $directory = 'public/documentos/documentos_fiscales';
                $fullPath = storage_path('app/' . $directory);

                // Crear directorio recursivamente si no existe usando File facade
                if (!File::exists($fullPath)) {
                    File::makeDirectory($fullPath, 0755, true, true);
                }

                $path = $request->documento_fiscal_file->store($directory);
                $validatedData['documento_fiscal'] = $path;
                // Eliminar el archivo anterior si existe
                if($datoFiscal->documento_fiscal) {
                    $oldPath = str_replace('public/', '', $datoFiscal->documento_fiscal);
                    if(Storage::disk('public')->exists($oldPath)) {
                        Storage::disk('public')->delete($oldPath);
                    }
                }
            }

            $datoFiscal->update($validatedData);

            DB::commit();

            return response()->json([
                'data' => $datoFiscal,
                'message' => 'Datos fiscales actualizados exitosamente.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Error al actualizar los datos fiscales.',
                'error' => $e->getMessage()
            ], JsonResponse::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    public function destroy(DatoFiscal $datoFiscal): JsonResponse
    {
        if ($datoFiscal->es_principal === 'si') {
            $otrasActivas = DatoFiscal::where('cliente_id', $datoFiscal->cliente_id)
                ->where('id', '!=', $datoFiscal->id)
                ->where('estatus', 'activo')
                ->count();
            if ($otrasActivas === 0) {
                return response()->json([
                    'message' => 'No se puede inactivar el único dato fiscal principal activo.'
                ], JsonResponse::HTTP_BAD_REQUEST);
            }
        }

        try {
            // Inactivar el registro en lugar de eliminarlo
            $datoFiscal->update(['estatus' => 'inactivo']);

            return response()->json([
                'message' => 'Dato fiscal inactivado exitosamente.',
                'data' => $datoFiscal
            ], JsonResponse::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error al inactivar los datos fiscales.',
                'error' => $e->getMessage()
            ], JsonResponse::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    public function getForSelect(Cliente $cliente): JsonResponse
    {
        $datosFiscales = DatoFiscal::where('cliente_id', $cliente->id)
            ->where('estatus', 'activo')
            ->orderBy('es_principal', 'desc')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($datoFiscal) {
                $datoFiscal->documento_fiscal_path = $datoFiscal->documento_fiscal
                    ? asset(str_replace("public", "storage", $datoFiscal->documento_fiscal))
                    : null;
                return $datoFiscal;
            });

        return response()->json(['data' => $datosFiscales]);
    }
}
