<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

return new class extends Migration
{
    /**
     * Run the migrations.
     * 
     * Sistema de Control de Lavado - Permisos y roles específicos del módulo de lavandería
     * Asigna permisos a los roles: Administrador, Recepcionista y Operador
     */
    public function up(): void
    {
        // Crear permisos específicos del módulo de lavandería del Sistema de Control de Lavado
        $permisosLavanderia = [
            'lavanderia.recepcion',    // Permiso para recepción de órdenes
            'lavanderia.operador',     // Permiso para operadores de áreas (Lavado, Secado, Planchado)
            'lavanderia.dashboard',    // Permiso para ver el dashboard semáforo
            'lavanderia.consultar',    // Permiso para consultar órdenes
        ];

        foreach ($permisosLavanderia as $permiso) {
            Permission::firstOrCreate([
                'name' => $permiso,
                'guard_name' => 'web'
            ]);
        }

        // Crear roles específicos del Sistema de Control de Lavado (si no existen)
        $recepcionistaRole = Role::firstOrCreate([
            'name' => 'Recepcionista',
            'guard_name' => 'web'
        ]);

        $operadorRole = Role::firstOrCreate([
            'name' => 'Operador',
            'guard_name' => 'web'
        ]);

        // Asignar permisos al rol Recepcionista del Sistema de Control de Lavado
        // Puede: registrar órdenes, procesar cobros, entregar ropa, consultar órdenes, ver dashboard
        $recepcionistaRole->givePermissionTo([
            'lavanderia.recepcion',
            'lavanderia.consultar',
            'lavanderia.dashboard',
        ]);

        // Asignar permisos al rol Operador del Sistema de Control de Lavado
        // Puede: operar en áreas de proceso (Lavado, Secado, Planchado), ver dashboard
        $operadorRole->givePermissionTo([
            'lavanderia.operador',
            'lavanderia.dashboard',
        ]);

        // Asignar todos los permisos de lavandería al rol Administrador del Sistema de Control de Lavado
        // El Administrador tiene acceso completo a todas las funcionalidades
        $adminRole = Role::where('name', 'Administrador')->first();
        if ($adminRole) {
            $adminRole->givePermissionTo($permisosLavanderia);
        }
    }

    /**
     * Reverse the migrations.
     * 
     * Elimina los permisos del módulo de lavandería del Sistema de Control de Lavado
     */
    public function down(): void
    {
        // Eliminar permisos del módulo de lavandería
        $permisosLavanderia = [
            'lavanderia.recepcion',
            'lavanderia.operador',
            'lavanderia.dashboard',
            'lavanderia.consultar',
        ];
        
        Permission::whereIn('name', $permisosLavanderia)->delete();

        // Nota: No eliminamos los roles Recepcionista y Operador porque son parte del Sistema de Control de Lavado
    }
};
